/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.license;

import chemaxon.license.TelemetryJsonUtil;
import chemaxon.license.audit.LicenseCheckResult;
import chemaxon.license.audit.LicenseUsageEntry;
import chemaxon.license.audit.UserDataEntry;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONObject;

class TelemetryFileHandler {
    private static final String TEMP_FILE_EXTENSION = ".temp";
    static Logger telemetryLogger = Logger.getLogger("cxl.license.telemetry");
    private final Path path;
    private final Path closedPath;
    private final Path archivedPath;

    public TelemetryFileHandler(String auditFilePath, String auditFileName) {
        this.path = auditFilePath == null || auditFileName == null ? null : Path.of(auditFilePath, auditFileName);
        this.closedPath = auditFilePath == null || auditFileName == null ? null : Path.of(auditFilePath, "closed_" + auditFileName);
        this.archivedPath = auditFilePath == null || auditFileName == null ? null : Path.of(auditFilePath, "archived_" + auditFileName);
        this.tryCloseAuditFile();
    }

    public synchronized void write(JSONObject jsonObject) throws IOException, InterruptedException {
        this.write(jsonObject, 0);
    }

    private synchronized boolean write(JSONObject jsonObject, int reTry) throws IOException, InterruptedException {
        if (this.path == null) {
            telemetryLogger.log(Level.INFO, "Can not write telemetry log to file because path is null.");
            return false;
        }
        telemetryLogger.log(Level.FINER, () -> "Write with " + reTry + ". retry data: " + jsonObject.toString());
        try (FileWriter writer = new FileWriter(this.path.toFile(), true);){
            writer.write(String.format("%s%n", jsonObject.toString()));
        }
        catch (FileNotFoundException e) {
            if (reTry < 3) {
                TimeUnit.MILLISECONDS.sleep(50L);
                return this.write(jsonObject, reTry + 1);
            }
            telemetryLogger.log(Level.INFO, "Can not write telemetry log to file.");
            return false;
        }
        return true;
    }

    public synchronized Collection<LicenseUsageEntry> getLicenseUsageEntriesAndArchive(String actualSession, int limit) {
        telemetryLogger.log(Level.FINER, () -> "Get license usage entries from audit file.");
        if (this.closedPath == null || this.archivedPath == null) {
            telemetryLogger.log(Level.FINER, () -> "Can not get license usage entries from audit file becauseclosed or archived audit path is null. Closed path: " + this.closedPath + " , archivedPath: " + this.archivedPath);
            return List.of();
        }
        HashMap<String, LicenseUsageEntry> resultLines = new HashMap<String, LicenseUsageEntry>();
        File closedFile = this.closedPath.toFile();
        File tempFile = new File(closedFile + TEMP_FILE_EXTENSION);
        boolean archived = false;
        if (this.isClosedFileExist()) {
            telemetryLogger.log(Level.FINER, () -> "Closed audit file exist, get license usage entries from that file.");
            try (BufferedReader reader = new BufferedReader(new FileReader(closedFile));
                 BufferedWriter tempWriter = new BufferedWriter(new FileWriter(tempFile, false));
                 BufferedWriter archivedWriter = new BufferedWriter(new FileWriter(this.archivedPath.toFile(), true));){
                String line;
                while ((line = reader.readLine()) != null) {
                    if (line.contains(actualSession)) {
                        throw new IllegalStateException("Closed telemetry file contains records from actual session.");
                    }
                    if (!line.contains("licenseKey")) {
                        archivedWriter.write(line);
                        archivedWriter.newLine();
                        archived = true;
                        continue;
                    }
                    if (this.isLicenseUsageLine(line)) {
                        this.mergeLicenseUsageIfUnderLimit(resultLines, TelemetryJsonUtil.getLicenseUsages(new JSONObject(line)), limit);
                        tempWriter.write(line);
                        tempWriter.newLine();
                        continue;
                    }
                    tempWriter.write(line);
                    tempWriter.newLine();
                }
            }
            catch (IOException e) {
                telemetryLogger.log(Level.INFO, "Can not get LicenseUsageEntries from closed audit file.", e);
                return List.of();
            }
            this.handleFiles(archived, closedFile, tempFile);
        } else {
            telemetryLogger.log(Level.FINER, () -> "Can not get license usage entries from audit file, because closedaudit file not exist.");
        }
        return resultLines.values();
    }

    private void handleFiles(boolean changedClosedFile, File closedFile, File tempFile) {
        telemetryLogger.log(Level.FINER, () -> "Clean telemetry files after operation.");
        if (changedClosedFile) {
            telemetryLogger.log(Level.FINER, () -> "Closed audit file changed by operation, delete closed audit fileand rename temp audit file to closed audit file, if contains any audit record");
            this.deleteClosedFiles(closedFile);
            this.deleteOrRenameTempFile(tempFile, closedFile);
        } else {
            telemetryLogger.log(Level.FINER, () -> "Closed audit file haven't changed, delete temp audit file.");
            this.deleteTempFile(tempFile);
        }
    }

    private void deleteOrRenameTempFile(File tempFile, File closedFile) {
        if (this.isNotEmptyFile(tempFile)) {
            this.renameFile(tempFile, closedFile);
        } else {
            this.deleteTempFile(tempFile);
        }
    }

    private void renameFile(File from, File to) {
        if (from.renameTo(to)) {
            telemetryLogger.log(Level.FINER, () -> "File: " + from + " successfully renamed to: " + to);
        } else {
            telemetryLogger.log(Level.FINER, () -> "File: " + from + " can not be renamed to: " + to);
        }
    }

    private void deleteClosedFiles(File closedFile) {
        telemetryLogger.log(Level.FINER, () -> "Delete closed audit file.");
        if (this.deleteFile(closedFile)) {
            telemetryLogger.log(Level.FINER, () -> "Closed audit file successfully deleted.");
        }
    }

    private void deleteTempFile(File tempFile) {
        telemetryLogger.log(Level.FINER, () -> "Delete temp audit file.");
        if (this.deleteFile(tempFile)) {
            telemetryLogger.log(Level.FINER, () -> "Temp closed audit file successfully deleted.");
        }
    }

    private boolean deleteFile(File file) {
        try {
            Files.delete(file.toPath());
            return true;
        }
        catch (Exception e) {
            telemetryLogger.log(Level.FINER, e, () -> "Audit file can not be deleted. Path: " + file);
            return false;
        }
    }

    private boolean isLicenseUsageLine(String line) {
        return line.contains("licenseUsage");
    }

    private void mergeLicenseUsageIfUnderLimit(Map<String, LicenseUsageEntry> resultLines, LicenseUsageEntry licenseUsage, int limit) {
        if (resultLines.containsKey(licenseUsage.sessionId()) || resultLines.size() < limit) {
            resultLines.compute(licenseUsage.sessionId(), (k, v) -> v == null ? licenseUsage : this.merge((LicenseUsageEntry)v, licenseUsage));
        }
    }

    private LicenseUsageEntry merge(LicenseUsageEntry olderValue, LicenseUsageEntry newerValue) {
        Map mergeLicenseUsage = olderValue.licenseUsage();
        newerValue.licenseUsage().forEach((key, value) -> mergeLicenseUsage.compute(key, (k, v) -> v == null ? value : this.mergeCheckResult((LicenseCheckResult)value, (LicenseCheckResult)mergeLicenseUsage.get(key))));
        return olderValue;
    }

    private LicenseCheckResult mergeCheckResult(LicenseCheckResult value1, LicenseCheckResult value2) {
        return new LicenseCheckResult(0L, value1.getSuccess() + value2.getSuccess(), value1.getFail() + value2.getFail());
    }

    private synchronized void tryCloseAuditFile() {
        try {
            telemetryLogger.log(Level.FINER, () -> "Try close audit file");
            if (this.path == null || this.closedPath == null) {
                telemetryLogger.log(Level.FINER, () -> "Can not close audit file, because path or closedPath is null.Path: " + this.path + " ClosedPath: " + this.closedPath);
                return;
            }
            File closedAuditFile = this.closedPath.toFile();
            File actualAuditFile = this.path.toFile();
            if (!closedAuditFile.exists() && actualAuditFile.exists() && this.isNotEmptyFile(actualAuditFile)) {
                this.renameFile(actualAuditFile, closedAuditFile);
            }
        }
        catch (Exception e) {
            telemetryLogger.log(Level.INFO, e, () -> "Can not close audit log file.");
        }
    }

    private boolean isClosedFileExist() {
        if (this.closedPath == null) {
            return false;
        }
        File closedAuditFile = this.closedPath.toFile();
        return closedAuditFile.exists();
    }

    private boolean isNotEmptyFile(File actualAuditFile) {
        return actualAuditFile.length() > 0L;
    }

    public void tryDeleteLicenseUsageRecords(Iterable<String> sessionIds) {
        this.tryDeleteRecords(sessionIds, this::isLicenseUsageLine);
    }

    public void tryDeleteUserDataRecords(Iterable<String> sessionIds) {
        this.tryDeleteRecords(sessionIds, this::isUserDataLine);
    }

    private void tryDeleteRecords(Iterable<String> sessionIds, Predicate<String> lineTypeTester) {
        ArrayList<String> sessionIdList = new ArrayList<String>();
        sessionIds.forEach(sessionIdList::add);
        if (sessionIdList.isEmpty() || this.closedPath == null) {
            return;
        }
        this.deleteLinesFromFile(sessionIdList, lineTypeTester);
    }

    private synchronized void deleteLinesFromFile(List<String> sessionIds, Predicate<String> lineTypeTester) {
        File closedFile = this.closedPath.toFile();
        if (!closedFile.exists()) {
            throw new UnsupportedOperationException("Closed audit file not exist.");
        }
        File tempFile = new File(closedFile + TEMP_FILE_EXTENSION);
        try (BufferedReader reader = new BufferedReader(new FileReader(closedFile));
             BufferedWriter writer = new BufferedWriter(new FileWriter(tempFile, false));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (lineTypeTester.test(line) && this.isDeletableLineBySessionId(line, sessionIds)) continue;
                writer.write(line);
                writer.newLine();
            }
        }
        catch (IOException e) {
            return;
        }
        this.handleFiles(true, closedFile, tempFile);
    }

    private boolean isUserDataLine(String line) {
        return line.contains("cclVersion");
    }

    private boolean isDeletableLineBySessionId(String line, List<String> sessionIds) {
        return sessionIds.stream().anyMatch(line::contains);
    }

    public synchronized Iterable<UserDataEntry> getUserDataEntriesAndArchive(int limit) {
        if (this.closedPath == null || this.archivedPath == null) {
            return List.of();
        }
        ArrayList<UserDataEntry> userDataEntries = new ArrayList<UserDataEntry>();
        File closedFile = this.closedPath.toFile();
        File tempFile = new File(closedFile + TEMP_FILE_EXTENSION);
        boolean archived = false;
        if (this.isClosedFileExist()) {
            try (BufferedReader reader = new BufferedReader(new FileReader(closedFile));
                 BufferedWriter tempWriter = new BufferedWriter(new FileWriter(tempFile, false));
                 BufferedWriter archivedWriter = new BufferedWriter(new FileWriter(this.archivedPath.toFile(), true));){
                String line;
                while ((line = reader.readLine()) != null) {
                    if (!line.contains("licenseKey")) {
                        archivedWriter.write(line);
                        archivedWriter.newLine();
                        archived = true;
                        continue;
                    }
                    if (line.contains("cclVersion") && userDataEntries.size() < limit) {
                        userDataEntries.add(TelemetryJsonUtil.getUserData(new JSONObject(line)));
                        tempWriter.write(line);
                        tempWriter.newLine();
                        continue;
                    }
                    tempWriter.write(line);
                    tempWriter.newLine();
                }
            }
            catch (IOException e) {
                telemetryLogger.log(Level.INFO, "Can not get UserDataEntries from closed audit file.", e);
                return List.of();
            }
            this.handleFiles(archived, closedFile, tempFile);
        }
        return userDataEntries;
    }
}

